<?php
/*--------------------------------------------------------------
   GetWidgetAction.php 2021-08-26
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Action;

use Exception;
use Gambio\Admin\Modules\Language\Model\ValueObjects\LanguageCode;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option\Dropdown as DropdownOption;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option\Dropdown\Item as DropdownItem;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Map as MapData;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Map\Item as MapItem;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Serial as SerialData;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Serial\Category as SerialCategory;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Serial\Item as SerialItem;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Serial\Item\Value as SerialItemValue;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Table as TableData;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Table\Column as TableColumn;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Table\Column\Date as DateTableColumn;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Table\Row as TableRow;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Table\Row\Field as TableRowField;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Text as TextData;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data\Number as NumberData;
use Gambio\Core\Application\Http\AbstractAction;
use Gambio\Core\Application\ValueObjects\UserPreferences;
use Gambio\Core\Language\App\LanguageService;
use InvalidArgumentException;

/**
 * @codeCoverageIgnore
 */
abstract class GetWidgetAction extends AbstractAction
{
    private const FALLBACK_LANGUAGE_CODE = "de";
    
    /**
     * @var LanguageCode
     */
    private $languageCode;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    /**
     * @var LanguageService
     */
    private $languageService;
    
    
    /**
     * Constructor.
     */
    public function __construct(LanguageService $languageService, UserPreferences $userPreferences)
    {
        $this->languageService = $languageService;
        $this->userPreferences = $userPreferences;
        
        try {
            $this->languageCode = $this->languageService->getLanguageById($this->userPreferences->languageId())->code();
        } catch (Exception $exception) {
            $this->languageCode = self::FALLBACK_LANGUAGE_CODE;
        }
    }
    
    
    /**
     * Return data as array.
     */
    protected function mapDataToArray(Data $data): array
    {
        if ($data instanceof SerialData) {
            return $this->mapSerialDataToArray($data);
        }
        
        if ($data instanceof MapData) {
            return $this->mapMapDataToArray($data);
        }
        
        if ($data instanceof TableData) {
            return $this->mapTableDataToArray($data);
        }
        
        if ($data instanceof TextData) {
            return $this->mapTextDataToArray($data);
        }
        
        if ($data instanceof NumberData) {
            return $this->mapNumberDataToArray($data);
        }
        
        throw new InvalidArgumentException("Unknown data type");
    }
    
    
    /**
     * Return array containing serial data.
     */
    protected function mapSerialDataToArray(SerialData $data): array
    {
        $array = [
            "type" => $data->type()
        ];
        
        $array["categories"] = array_map(function (SerialCategory $category) {
            return $category->value();
        },
            iterator_to_array($data->categories()));
        
        $array["series"] = array_map(function (SerialItem $series) {
            return [
                "title"  => $series->title()->value(),
                "values" => array_map(function (SerialItemValue $value) {
                    return $value->value();
                },
                    iterator_to_array($series->values()))
            ];
        },
            iterator_to_array($data->series()));
        
        return $array;
    }
    
    
    /**
     * Return array containing map data.
     */
    protected function mapMapDataToArray(MapData $data): array
    {
        $array = [
            "type" => $data->type()
        ];
        
        $array["values"] = array_map(function (MapItem $map) {
            return [
                "title" => $map->title()->value(),
                "value" => $map->value()->value()
            ];
        },
            iterator_to_array($data->mapItems()));
        
        return $array;
    }
    
    
    /**
     * Return array containing table data.
     */
    protected function mapTableDataToArray(TableData $data): array
    {
        $array = [
            "type" => $data->type()
        ];
        
        $array["columns"] = array_map(function (TableColumn $column) {
            $data = [
                "title" => $column->title(),
                "field" => $column->field(),
                "type"  => $column->type()
            ];
            
            if ($column instanceof DateTableColumn) {
                $data["format"] = [
                    "input"  => $column->inputDateFormat(),
                    "output" => $column->outputDateFormat()
                ];
            }
            
            return $data;
        },
            iterator_to_array($data->columns()));
        
        $array["rows"] = array_map(function (TableRow $row) {
            $fields = [];
            
            /**
             * @var TableRowField $field
             */
            foreach ($row->fields() as $field) {
                $fields[$field->name()] = $field->value();
            }
            
            return $fields;
        },
            iterator_to_array($data->rows()));
        
        return $array;
    }
    
    
    /**
     * Return array containing text data.
     */
    protected function mapTextDataToArray(TextData $data): array
    {
        return [
            "type"  => $data->type(),
            "value" => $data->text()->value()
        ];
    }
    
    
    /**
     * Return array containing number data.
     */
    protected function mapNumberDataToArray(NumberData $data): array
    {
        return [
            "type" => $data->type(),
            "value" => $data->number()->value()
        ];
    }
    
    
    /**
     * Return options as array.
     */
    protected function mapOptionsToArray(Options $options): array
    {
        $array = [];
        
        /**
         * @var Option $option
         */
        foreach ($options as $option) {
            $itemArray = [
                "type"  => $option->type(),
                "value" => $option->value(),
                "title" => $option->titles()->getByLanguageCode($this->languageCode)->value()
            ];
            
            if ($option instanceof DropdownOption) {
                $dropdownArray = [];
                
                /**
                 * @var DropdownItem $item
                 */
                foreach ($option->items() as $item) {
                    $dropdownArray[$item->value()->value()] = $item->titles()
                        ->getByLanguageCode($this->languageCode)
                        ->value();
                }
                
                $itemArray["dropdown"] = $dropdownArray;
            }
            
            $array[$option->id()->value()] = $itemArray;
        }
        
        return $array;
    }
}